/*
  ==============================================================================

    SonicCrypt Chaos Engine
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/
#pragma once

#include <JuceHeader.h>

enum class ChaosMode { OneShot, Rhythm, Drone };

// --- DATA STRUCTURES ---
class SonicCryptSound : public juce::SynthesiserSound
{
public:
    SonicCryptSound() {}
    bool appliesToNote(int) override { return true; }
    bool appliesToChannel(int) override { return true; }

    struct LayerData {
        juce::AudioBuffer<float> buffer;
        double sourceRate = 44100.0;
        bool isValid = false;
        float pan = 0.5f; // 0.0 = Left, 1.0 = Right
    };

    std::array<LayerData, 4> layers;

    void setLayer(int index, const juce::AudioBuffer<float>& data, double rate, float pan)
    {
        if (index >= 0 && index < 4) {
            layers[index].buffer.makeCopyOf(data);
            layers[index].sourceRate = rate;
            layers[index].pan = pan;
            layers[index].isValid = true;
        }
    }

    void clear() { for (auto& l : layers) l.isValid = false; }
};

class SonicCryptVoice : public juce::SynthesiserVoice
{
public:
    SonicCryptVoice() {}
    bool canPlaySound(juce::SynthesiserSound* sound) override { return dynamic_cast<SonicCryptSound*>(sound) != nullptr; }

    void startNote(int midiNoteNumber, float velocity, juce::SynthesiserSound* sound, int) override
    {
        if (auto* s = dynamic_cast<SonicCryptSound*>(sound))
        {
            adsr.setSampleRate(getSampleRate());
            adsr.noteOn();
            double pitchRatio = std::pow(2.0, (midiNoteNumber - 60) / 12.0);

            for (int i = 0; i < 4; ++i) {
                if (s->layers[i].isValid) {
                    playStates[i].active = true;
                    playStates[i].pos = 0.0;
                    playStates[i].inc = (s->layers[i].sourceRate / getSampleRate()) * pitchRatio;
                    float p = s->layers[i].pan;
                    playStates[i].gainL = std::cos(p * juce::MathConstants<float>::halfPi);
                    playStates[i].gainR = std::sin(p * juce::MathConstants<float>::halfPi);
                }
                else { playStates[i].active = false; }
            }
        }
    }

    void stopNote(float velocity, bool allowTailOff) override {
        adsr.noteOff();
        if (!allowTailOff) { clearCurrentNote(); adsr.reset(); }
    }

    void renderNextBlock(juce::AudioBuffer<float>& outputBuffer, int startSample, int numSamples) override
    {
        if (!isVoiceActive()) return;
        auto* sound = dynamic_cast<SonicCryptSound*>(getCurrentlyPlayingSound().get());
        if (!sound) return;

        float* outL = outputBuffer.getWritePointer(0);
        float* outR = (outputBuffer.getNumChannels() > 1) ? outputBuffer.getWritePointer(1) : outL;

        for (int i = 0; i < numSamples; ++i)
        {
            float env = adsr.getNextSample();
            if (env <= 0.0001f && !adsr.isActive()) { clearCurrentNote(); break; }

            float sumL = 0.0f;
            float sumR = 0.0f;

            for (int L = 0; L < 4; ++L) {
                if (!playStates[L].active) continue;
                auto& layer = sound->layers[L];
                const float* inL = layer.buffer.getReadPointer(0);
                const float* inR = (layer.buffer.getNumChannels() > 1) ? layer.buffer.getReadPointer(1) : inL;
                int len = layer.buffer.getNumSamples();

                if (playStates[L].pos >= len - 1) {
                    if (shouldLoop) { while (playStates[L].pos >= len - 1) playStates[L].pos -= (len - 1); }
                    else { playStates[L].active = false; continue; }
                }

                double pos = playStates[L].pos;
                int idxA = (int)pos;
                int idxB = idxA + 1;
                float alpha = (float)(pos - idxA);
                if (idxB >= len) idxB = (shouldLoop ? 0 : idxA);

                float sampL = inL[idxA] + alpha * (inL[idxB] - inL[idxA]);
                float sampR = inR[idxA] + alpha * (inR[idxB] - inR[idxA]);

                sumL += sampL * playStates[L].gainL;
                sumR += sampR * playStates[L].gainR;

                playStates[L].pos += playStates[L].inc;
            }

            outL[startSample + i] += sumL * env * 0.3f;
            if (outputBuffer.getNumChannels() > 1)
                outR[startSample + i] += sumR * env * 0.3f;
        }
    }
    void pitchWheelMoved(int) override {}
    void controllerMoved(int, int) override {}
    void setLooping(bool b) { shouldLoop = b; }
    void setADSR(const juce::ADSR::Parameters& p) { adsr.setParameters(p); }

private:
    struct PlayState {
        bool active = false;
        double pos = 0.0;
        double inc = 1.0;
        float gainL = 0.7f;
        float gainR = 0.7f;
    };
    std::array<PlayState, 4> playStates;
    bool shouldLoop = true;
    juce::ADSR adsr;
};

// --- PROCESSOR ---
class SonicCryptChaosAudioProcessor : public juce::AudioProcessor, public juce::ValueTree::Listener
{
public:
    SonicCryptChaosAudioProcessor();
    ~SonicCryptChaosAudioProcessor() override;

    void prepareToPlay(double sampleRate, int samplesPerBlock) override;
    void releaseResources() override;
    bool isBusesLayoutSupported(const BusesLayout& layouts) const override;
    void processBlock(juce::AudioBuffer<float>&, juce::MidiBuffer&) override;

    juce::AudioProcessorEditor* createEditor() override;
    bool hasEditor() const override;

    const juce::String getName() const override;
    bool acceptsMidi() const override;
    bool producesMidi() const override;
    bool isMidiEffect() const override;
    double getTailLengthSeconds() const override;

    int getNumPrograms() override;
    int getCurrentProgram() override;
    void setCurrentProgram(int index) override;
    const juce::String getProgramName(int index) override;
    void changeProgramName(int index, const juce::String& newName) override;

    void getStateInformation(juce::MemoryBlock& destData) override;
    void setStateInformation(const void* data, int sizeInBytes) override;

    juce::AudioProcessorValueTreeState& getAPVTS() { return apvts; }
    void triggerRandomChaos();
    void loadSamplesFromFolder(const juce::File& folder);

    juce::File getCurrentSampleFolder() const { return currentSampleFolder; }
    juce::File getPresetFolder() const;

    void savePreset(const juce::String& name);
    void loadPresetFile(const juce::File& file);
    void resetEffects();

    std::atomic<bool> isLoading{ false };

    // VISUALIZATION DATA
    std::atomic<float> currentOutputLevel{ 0.0f };

    // --- UI PERSISTENCE ---
    int lastUIWidth = 1000;
    int lastUIHeight = 600;

private:
    juce::AudioProcessorValueTreeState apvts;
    juce::AudioProcessorValueTreeState::ParameterLayout createParameterLayout();
    void valueTreePropertyChanged(juce::ValueTree&, const juce::Identifier&) override;

    juce::Synthesiser synth;
    const int maxVoices = 8;
    juce::AudioFormatManager formatManager;
    juce::CriticalSection sampleLoadingLock;

    juce::dsp::LadderFilter<float> filterModule;
    juce::dsp::Reverb reverbModule;
    juce::dsp::Gain<float> gainModule;
    juce::dsp::WaveShaper<float> distortionModule;
    juce::dsp::DelayLine<float> delayModule{ 192000 };
    juce::dsp::Limiter<float> limiterModule;

    juce::SmoothedValue<float> delayTimeSmoothed;

    ChaosMode currentMode = ChaosMode::Rhythm;
    juce::File currentSampleFolder;
    juce::StringArray allSamplePaths;
    std::array<float, 16> rhythmPattern;
    float lfoPhase = 0.0f;
    double gatePhaseSamples = 0.0;
    int lastNote = 60;

    void shuffleSamplesFromCache();
    void loadSmartSliceIntoSound(SonicCryptSound* sound, int layerIndex, juce::File file);
    void writeAudioLayerToStream(juce::OutputStream& out, SonicCryptSound* sound, int layerIndex);
    void readAudioLayerFromStream(juce::InputStream& in, SonicCryptSound* sound, int layerIndex);

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(SonicCryptChaosAudioProcessor)
};